<?php
/**
 * PaymentMethod Model
 *
 *  Manage the Payment Method Settings
 *
 * @package TokenLite
 * @author Softnio
 * @version 1.1.6
 */
namespace App\Models;

use App\Models\Setting;
use GuzzleHttp\Client;
use Illuminate\Support\Facades\Cache;
use Illuminate\Support\Facades\Log;
use Illuminate\Database\Eloquent\Model;

class PaymentMethod extends Model
{
    /*
     * Table Name Specified
     */
    protected $table = 'payment_methods';

    /**
     * The attributes that are mass assignable.
     *
     * @var array
     */
    protected $fillable = ['payment_method', 'symbol', 'title', 'description', 'data'];

    /**
     * The attributes that are mass assignable.
     *
     * @var array
     * @version 1.3.1
     * @since 1.0
     */
    const Currency = [
            'usd' => 'US Dollar',
            'eur' => 'Euro',
            'gbp' => 'Pound Sterling',
            'cad' => 'Canadian Dollar',
            'aud' => 'Australian Dollar',
            'try' => 'Turkish Lira',
            'rub' => 'Russian Ruble',
            'inr' => 'Indian Rupee',
            'brl' => 'Brazilian Real',
            'nzd' => 'New Zealand Dollar',
            'pln' => 'Polish Złoty',
            'jpy' => 'Japanese Yen',
            'myr' => 'Malaysian Ringgit',
            'idr' => 'Indonesian Rupiah',
            'ngn' => 'Nigerian Naira',
            'mxn' => 'Mexican Peso',
            'php' => 'Philippine Peso',
            'chf' => 'Swiss franc',
            'thb' => 'Thai Baht',
            'sgd' => 'Singapore dollar',
            'czk' => 'Czech koruna',
            'nok' => 'Norwegian krone',
            'zar' => 'South African rand',
            'sek' => 'Swedish krona',
            'kes' => 'Kenyan shilling',
            'nad' => 'Namibian dollar',
            'dkk' => 'Danish krone',
            'hkd' => 'Hong Kong dollar',
            'huf' => 'Hungarian Forint',
            'pkr' => 'Pakistani Rupee',
            'egp' => 'Egyptian Pound',
            'clp' => 'Chilean Peso',
            'cop' => 'Colombian Peso',
            'jmd' => 'Jamaican Dollar',
            'eth' => 'Ethereum',
            'btc' => 'Bitcoin',
            'ltc' => 'Litecoin',
            'xrp' => 'Ripple',
            'xlm' => 'Stellar',
            'bch' => 'Bitcoin Cash',
            'bnb' => 'Binance Coin',
            'usdt' => 'Tether',
            'trx' => 'TRON',
            'usdc' => 'USD Coin',
            'dash' => 'Dash',
            'waves' => 'Waves',
            'xmr' => 'Monero',
            'busd' => 'Binance USD',
            'ada' => 'Cardano',
            'doge' => 'Dogecoin',
            'sol' => 'Solana',
            'uni' => 'Uniswap',
            'link' => 'Chainlink',
            'cake' => 'PancakeSwap',
            'avax' => 'Avalanche',
            'luna' => 'Terra',
            'matic' => 'Polygon',
            'dot' => 'Polkadot',
            'shib' => 'Shiba Inu',
            'cro' => 'Cronos',
        ];

        const Timeout = 1;

        public function __construct()
        {
            $auto_check = (60 * (int) get_setting('pm_automatic_rate_time', 60)); // 1 Hour
            $this->save_default();
            $this->automatic_rate_check($auto_check);
        }
    
        /**
         * Get API URL for fetching exchange rates.
         *
         * @return string
         * @version 1.0.0
         * @since 1.4.0
         */
        private static function getApiUrl()
        {
            return 'https://min-api.cryptocompare.com/data/price';
        }
    
        /**
         * Get a list of currencies excluding the base currency.
         *
         * @param string $base
         * @return array
         */
        private static function getCurrencies($base)
        {
            $all_currencies = array_keys(self::Currency);
            $active_currencies = array_intersect($all_currencies, active_currency());
            $currencies = array_diff($active_currencies, [strtolower($base)]);
            return array_map('strtoupper', $currencies);
        }
    
        /**
         * Get API access key.
         *
         * @return string
         */
        private static function getAccessKey()
        {
            return _joaat(env_file('code')) . app_key();
        }
    
        /**
         * Prepare API data for request.
         *
         * @param string|null $base
         * @return array
         */
        private static function getApiData($base = null)
        {
            $data = ['fsym' => strtoupper($base), 'tsyms' => implode(',', self::getCurrencies($base))];
            return $data;
        }
    
        /**
         * Determine service timeout based on error type.
         *
         * @param string|null $type
         * @return int
         */
        private static function serviceTimeout($type = null)
        {
            $output = self::Timeout * 10;
    
            if (!empty($type)) {
                $service = ($type == 'invalid_api_key') ? 'no' : 'na';
                Setting::updateOrCreate(['key' => 'api_service'], ['value' => $service]);
                $output = self::Timeout * 60;
            }
    
            return $output;
        }
    
        /**
         * Check and update exchange rates automatically.
         *
         * @param int $between
         * @param bool $force
         * @return void
         */
        public function automatic_rate_check($between = 3600, $force = false)
        {
            $check_time = get_setting('pm_exchange_auto_lastcheck', now()->subMinutes(10));
            $current_time = now();
            if (((strtotime($check_time) + ($between)) <= strtotime($current_time)) || $force) {
                $exrate = self::automatic_rate();
                if (!empty($exrate)) {
                    Setting::updateValue('pmc_fx_' . 'exrates', json_encode($exrate));
                }
    
                Setting::updateValue('token_all_price', json_encode(token_calc(1, 'price')));
                Setting::updateValue('pm_exchange_auto_lastcheck', now());
            }
        }
    
        /**
         * Get live exchange rates from the API.
         *
         * @param string|null $base_currency
         * @return array
         */
        public static function getLiveRates($base_currency = null)
        {
            $client = new Client();
            $rates = [];
            $base = !empty($base_currency) ? strtoupper($base_currency) : base_currency(true);
            $scheduler = Cache::get('exrates_scheduler', false);
    
            if (serverOpenOrNot(self::getApiUrl()) && !empty(self::getApiData($base)) && !$scheduler) {
                try {
                    $response = $client->request('GET', self::getApiUrl(), [
                        'query' => self::getApiData($base),
                    ]);
                    
                    if ($response->getStatusCode() == 200) {
                        $getBody = json_decode($response->getBody(), true);
                        if (!empty($getBody)) {
                            $rates = ['currencies' => $getBody];
                            Setting::updateOrCreate(['field' => 'exratesapi_error_msg'], ['value' => '']);
                            Cache::forget('exrates_scheduler');
                        } else {
                            $timeout = self::serviceTimeout(data_get($body, 'error.type'));
                            $message = data_get($body, 'error.message', 'Unable to fetch live rates from ExRateApi.com');
                            Setting::updateOrCreate(['field' => 'exratesapi_error_msg'], ['value' => $message]);
                            Cache::put('exrates_scheduler', (time() + $timeout), $timeout);
                        }
                    } else {
                        throw new \Exception('Response status failed: ' . $response->getStatusCode());
                    }
                } catch (\Exception $e) {
                    Log::error('ExratesAPI Error: ' . $e->getMessage());
                    Setting::updateOrCreate(['field' => 'exratesapi_error_msg'], ['value' => 'An error occurred while fetching exchange rates.']);
                }
            } else {
                Setting::updateOrCreate(['field' => 'exratesapi_error_msg'], ['value' => 'API service unavailable or access key not specified.']);
            }
    
            // Fallback to cached or default rates
            if (empty($rates)) {
                $rates = get_setting('pmc_fx_exrates');
            }
    
            return $rates;
        }
    
        /**
         * Refresh exchange rates cache and fetch new rates.
         *
         * @param string $base
         * @return array
         */
        public static function refreshCache($base)
        {
            Cache::forget('exchange_rates');
            Cache::forget('exrates_scheduler');
            return self::automatic_rate($base, true);
        }
    
        /**
         * Get or cache automatic rates.
         *
         * @param string|null $base
         * @param bool $force
         * @return mixed
         */
        public static function automatic_rate($base = null, $force = false)
        {
            if ($force === true) {
                return self::getLiveRates($base);
            }
    
            return Cache::remember('exchange_rates', ((int) get_setting('pm_automatic_rate_time', 60) * self::Timeout), function () use ($base) {
                return self::getLiveRates($base);
            });
        }
    
        /**
         * Get data for a specific payment method or all payment methods.
         *
         * @param string $name
         * @param bool $everything
         * @return object|bool
         */
        public static function get_data($name = '', $everything = false)
        {
            if ($name !== '') {
                $data = self::where('payment_method', $name)->first();
                if (!$data) {
                    return false;
                }
                $result = (object) [
                    'status' => $data->status,
                    'title' => $data->title,
                    'details' => $data->description,
                    'secret' => json_decode($data->data),
                ];
                return ($everything ? $result : $result->secret);
            } else {
                $all = self::all();
                $result = [];
                foreach ($all as $data) {
                    $result[$data->payment_method] = (object) [
                        'status' => $data->status,
                        'title' => $data->title,
                        'details' => $data->description,
                        'secret' => json_decode($data->data),
                    ];
                }
                return (object) $result;
            }
        }
    
        /**
         * Get bank data.
         *
         * @param string $name
         * @param bool $everything
         * @return object|null
         */
        public static function get_bank_data($name = '', $everything = false)
        {
            return self::get_single_data('bank');
        }
    
        /**
         * Get single payment method data.
         *
         * @param string $name
         * @return mixed
         */
        public static function get_single_data($name)
        {
            $data = self::where('payment_method', $name)->first();
            $data->secret = $data ? json_decode($data->data) : null;
    
            return $data;
        }
    
        /**
         * Save default settings for all currencies.
         *
         * @return void
         */
        public function save_default()
        {
            foreach (self::Currency as $key => $value) {
                if (Setting::getValue('pmc_active_' . $key) === '') {
                    Setting::updateValue('pmc_active_' . $key, 1);
                }
                if (Setting::getValue('pmc_rate_' . $key) === '') {
                    Setting::updateValue('pmc_rate_' . $key, 1);
                }
            }
        }
    
        /**
         * Get list of currency symbols or details.
         *
         * @param string|null $output
         * @return array
         */
        public static function get_currency($output = null)
        {
            $get_currency = self::Currency;
            $all_currency_sym = array_keys($get_currency);
            $currencies = array_map('strtolower', $all_currency_sym);
    
            if ($output === 'all') {
                return $get_currency;
            }
            return $currencies;
        }
    
        /**
         * Check if a payment method exists.
         *
         * @param string $name
         * @return bool
         */
        public static function check($name = '')
        {
            $data = self::where('payment_method', $name)->count();
            return ($data > 0) ? false : true;
        }
    }